/************************************************************************
 * \file: trace_fe.c
 *
 * \version: $Id: trace_fe.c,v 1.27 2012/05/28 10:29:41 jayanth.mc Exp $
 *
 * This file implements the basic initialization and uninitialization for
 * Trace component.
 *
 * \component: Gen2 Trace
 *
 * \author B. Das         bhaskar.das(o)in.bosch.com
 *         Sakthivelu S.  sakthivelu.s(o)in.bosch.com
 * \modified: Arun V
 * \copyright: (c) 2003 - 2009 ADIT
 *
 ***********************************************************************/

#include "trace_base.h"
#include "pfcfg_interface.h"
#include <string.h>
#include <sys/prctl.h>

/* Version string */
EXPORT U8 g_TRACE_ver[] = TRACE_VERSION;

/* Trace manager */
EXPORT TRACE_mgr* g_TRACE_mgr = NULL;
/*SWGIII-3621: TRACE static constructor for init() dangerous */
EXPORT TRACE_start_fp TRACE_init_fp = TRACE_start;

/**
 * Create Trace task and bind to desired CPU core.
 *
 * \parm task   Task Entry point
 * \parm data   task data structure
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
EXPORT ER TRACE_create_task(pthread_t* tid, TRACE_task task, TRACE_mgr* mgr);
EXPORT ER TRACE_create_task(pthread_t* tid, TRACE_task task, TRACE_mgr* mgr)
{
    ER     rc           = E_OK;
    S32    stage        = 0;
    U32    stackSize    = TRACE_TASK_STACK_SIZE;
    VP id               = NULL;/*channel  stub task stack-size configurable @aba5kor*/
    pthread_attr_t attr;
    memset(&attr, 0, sizeof(attr));

    while((stage < 3) && (rc == E_OK))
    {
        switch(stage)
        {
            case 0:
                rc = pthread_attr_init(&attr);
            break;

            case 1:
                id = PFCFG_opn_cnf("adit_trace", PFCFG_READ);/*channel  stub task stack-size configurable @aba5kor*/

                if(id){
                          PFCFG_get_cfn(id, "TRACE_CHNL_STUB_TSK_STKSZ", (VP)&stackSize, 1);
                          TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "stackSize[%d]", stackSize);
                           
                          if (stackSize != 0) {
                                                 rc = pthread_attr_setstacksize(&attr,(size_t)stackSize);

                                              } 
                          else { 
                                  stackSize    = TRACE_TASK_STACK_SIZE;
                                  rc = pthread_attr_setstacksize(&attr,(size_t)stackSize);
                                }
                          
                          PFCFG_close_cnf(id);

                        }
  

                  else {   
                          stackSize    = TRACE_TASK_STACK_SIZE;
                          rc = pthread_attr_setstacksize(&attr,(size_t)stackSize);
                       }

            break;

            case 2:
                rc = pthread_create(tid, &attr, task, (VP)mgr);
            break;

            default:
            break;
        }   
        stage++;
    }
    if(stage > 1)
    {
        /* destroy only if attr init is success */
        rc = pthread_attr_destroy(&attr);
    }
    return rc;
}


LOCAL void TRACE_chnl_stub_cleanuphdlr(void* handler);
LOCAL void TRACE_chnl_stub_cleanuphdlr(void* handler)
{
  handler = handler;
  /* socket is closed in shutdown sequence */
  /* FixMe: any other cleanup code, if any */
}

LOCAL void TRACE_notify_stub_cleanuphdlr(void* handler);
LOCAL void TRACE_notify_stub_cleanuphdlr(void* handler)
{
  handler = handler;
  /* socket is closed in shutdown sequence */
  /* FixMe: any other cleanup code, if any */
}


LOCAL void* TRACE_chnl_stubtsk(void* exinf);
LOCAL void* TRACE_chnl_stubtsk(void* exinf)
{
  TRACE_mgr*           mgr   = (TRACE_mgr*)exinf;
  TRACE_callback_mgr* cmgr   = NULL;
  void*              payld   = NULL;
  S32           cur_chn_id   = 0;
  S32           rqst_chnl_id = 0;
  ER                  rc     = E_OK; 
  U8              bContinue  = TRUE;
  U32                 sz     = (mgr->devconf->lrg_pkt_sz + 
                               (S32)TRACE_CALBK_LEN_SZ  + 
                               (S32)1 /* chksum */      +
                               (S32)1 /* chnl_id */); 
  pid_t prc_id               = getpid();
  char prc_name_buf[32]      = {0};
  TRACE_status_msg st_msg    = {0, {0}};
  volatile U8 st_len         = sizeof(st_msg.status);
  
  sprintf(prc_name_buf, "TrcChTsk_%d%c", prc_id, '\0');
  prctl(PR_SET_NAME, prc_name_buf, 0, 0, 0);

  /* Clean up handler for thread clean up stack */
  pthread_cleanup_push(TRACE_chnl_stub_cleanuphdlr, NULL);
  
  TRACE_socket_init(&mgr->chan.stub.socket);
  /* SWGIII-7042 Fix*/
  rc = TRACE_socket_open_receiver(&mgr->chan.stub.socket,(S8*)mgr->sh->trc.d_cnfg.chnl_skt_loc, TRUE);
  if(E_OK == rc)
  {
    while(bContinue)
    {
       /* test if we are supposed to cancel */
       pthread_testcancel();

       /* wait for the message */
       TRACE_socket_receive(&mgr->chan.stub.socket, mgr->chan.lrg_pkt, sz);
      
       /* get the chan id */
	   /*@cjh1kor - Begin (For Channel Extension protocol Implementaion)*/
	  
	   
	   if( (S32)mgr->chan.lrg_pkt[0] == TR_CHN_EXTN_IDENTIFICATION )
	   {
		rqst_chnl_id = (S32)(mgr->chan.lrg_pkt[1] << TRACE_SHIFT_8bits) | (mgr->chan.lrg_pkt[2]); 
		cmgr = &mgr->chan.cmgr[rqst_chnl_id];
		cur_chn_id = cmgr->chn_id;
	   }
		/*@cjh1kor - End (For Channel Extension protocol Implementaion)*/
	   else
	   {
		rqst_chnl_id = (S32)mgr->chan.lrg_pkt[0]; 
		cmgr = &mgr->chan.cmgr[rqst_chnl_id];
		cur_chn_id = cmgr->chn_id;
	   }

	   /* get paylod */
	   /*@cjh1kor - Begin (For Channel Extension protocol Implementaion)*/
	   if( (S32)mgr->chan.lrg_pkt[0] == TR_CHN_EXTN_IDENTIFICATION )
	   {
		payld = &mgr->chan.lrg_pkt[3];
	   }
	   /*@cjh1kor - End (For Channel Extension protocol Implementaion)*/
	   else
	   {
		payld = &mgr->chan.lrg_pkt[1];
	   }
       rc = (ER)UTIL_test_swap32(&cmgr->chn_id, cmgr->chn_id, cmgr->chn_id); 
       
       if((rc == TRUE) && (cur_chn_id == rqst_chnl_id) && (NULL != cmgr->fp) ) /* Fix for Exception aus trace Bug */
       {
          /* invoke callback */
          (cmgr->fp)(payld);
       }
       else
       {
          /* send error message to TTFis */
          st_msg.status = (U8)TRACE_NO_SPT;
          st_msg.payld[TRACE_INDEX_0] = (U8)TRACE_CHAN_NOT_REG;
          st_len += TRACE_INDEX_1;
          (void)TRACE_snd_status(mgr,(U8*)(VP)&st_msg,st_len);
          if(NULL == cmgr->fp)
          {
            TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Callback Fn pointer is NULL for channel ID = 0x%x\n", cur_chn_id);
          }
       }
    }
  }
  else
  {
         /*Improvement SWGIII-7042 Fix: Adding fatal syslogs */
	  TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Trace ERROR : socket failure \n");
  }
  pthread_cleanup_pop(1);  
  return NULL;
}

LOCAL void* TRACE_notify_stubtsk(void* exinf);
LOCAL void* TRACE_notify_stubtsk(void* exinf)
{
  TRACE_mgr*           mgr   = (TRACE_mgr*)exinf; 
  TRACE_notify_inf*   inf    = NULL;
  ER                  rc     = E_OK; 
  U8              bContinue  = TRUE; 
  S32           default_idx  = 0;
  TRACE_evt_data   data      = {0, 0, 0, 0};

  /* Clean up handler for thread clean up stack */
  pthread_cleanup_push(TRACE_notify_stub_cleanuphdlr, NULL);
  
  TRACE_socket_init(&mgr->notify[default_idx].stub.socket);
  /* SWGIII-7042 Fix*/
  rc = TRACE_socket_open_receiver(&mgr->notify[default_idx].stub.socket, 
                                  (S8*)mgr->sh->trc.d_cnfg.chnl_skt_ntfy_loc, TRUE);
  if(E_OK == rc)
  {
    while(bContinue)
    {
       /* test if we are supposed to cancel */
       pthread_testcancel();

       /* wait for the message */
       TRACE_socket_receive(&mgr->notify[default_idx].stub.socket, 
                                   (S8*)&data, sizeof(data));
      
       inf = g_TRACE_mgr->notify[data.evt].inf;

       if((UTIL_test_swap32(&inf[data.idx].stat, 0x01, 0x01) == TRUE) && 
                                                  (inf[data.idx].fp != NULL))
       {
         /* invoke callback */
           (inf[data.idx].fp)((VP)&data.payld);
       }
       else
       {
           /* Error: event not registered */
       }
    }
  } 
  pthread_cleanup_pop(1);  
  return NULL;
}



/**
 * External interface for checking class/level status
 *
 * \parm  trClass        Trace class
 *        enLevel        Trace level
 * return \li BOOL       TR_ENABLED_CLASS_TRC, if enabled for TTFis
 *                       TR_ENABLED_CLASS_PXY, if enabled for Proxy
 *                       TR_ENABLED_CLASS_ALL, if enabled for all above 
 *                       TR_ENABLED_CLASS_NONE, otherwise
 */
LOCAL BOOL TR_core_bIsClassSelected2(U16 trClass, TR_tenTraceLevel enLevel);
LOCAL BOOL TR_core_bIsClassSelected2(U16 trClass, TR_tenTraceLevel enLevel)
{
  TRACE_clas_ena_stat tr_stat  = TR_ENABLED_CLASS_NONE;
  U8                  compid   = (U8)TR_COMPID(trClass) ;
  U8                  clasid   = (U8)TR_CLASID(trClass) ;
  TRACE_lvl           lvl;
  BOOL                result   = FALSE;

  memset(&lvl, 0, sizeof(TRACE_lvl));

/*SWGIII-3621: TRACE static constructor for init() dangerous */
  result = TRACE_init_fp();
  if(result >= E_OK)
  {
       if(TR_LEVEL_FATAL == enLevel) 
       {
         /* Fatal messages and PMM messages are always traced */
          tr_stat = TR_ENABLED_CLASS_ALL;
       }
       else
       {
          /* Enabled for TTFis */
          if((g_TRACE_mgr != NULL) &&
            (g_TRACE_mgr->c_cnfg->comp[compid].max_clas > 0) &&
             (clasid <= g_TRACE_mgr->c_cnfg->comp[compid].max_clas))
          {
              lvl = g_TRACE_mgr->c_cnfg->comp[compid].lvl[clasid];
          }

    tr_stat = (enLevel <= (TR_tenTraceLevel)lvl.trc_lvl) ?
          TR_ENABLED_CLASS_TRC : TR_ENABLED_CLASS_NONE;
    
    tr_stat = (TRACE_clas_ena_stat)((U32)tr_stat | 
                            (U32)((enLevel <= (TR_tenTraceLevel)lvl.pxy_lvl) ?
                            TR_ENABLED_CLASS_PXY : TR_ENABLED_CLASS_NONE));
      }
        result = (TR_ENABLED_CLASS_NONE == tr_stat) ? FALSE : (S32)tr_stat;
  }
 /*SWGIII-3621: TRACE static constructor for init() dangerous */
  else
  {
      TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "TR_core_bIsClassSelected2:TRACE_init failed\n");
      result = FALSE;
  }
  return result;
}



LOCAL void TRACE_cre_mpos_hdr(U32 len, U16 clas, U8 lvl, VP hedr,
                             U8 msgtyp, U8 packtyp);
LOCAL void TRACE_cre_mpos_hdr(U32 len, U16 clas, U8 lvl, VP hedr,
                             U8 msgtyp, U8 packtyp)
{
  TRACE_OUT_HDR* hdr = hedr;
  hdr->core_no       = UTIL_get_cpu_core();
  hdr->len           = len + sizeof(TRACE_OUT_HDR) +
                       sizeof(TRACE_MULTI_PACK_ID);
  hdr->proc_id       = 0;
  hdr->multi_pack    = msgtyp & TRACE_LOW_NIBBLE;
  hdr->pack_typ      = packtyp & TRACE_LOW_NIBBLE;
  hdr->comp_id       = TR_COMPID(clas);
  hdr->class_id      = TR_CLASID(clas);
  hdr->level         = lvl;
  hdr->state_byte    = 0;
  hdr->rsvd          = 0;
}

LOCAL void TRACE_cre_mpo_hdr(U32 len, TRACE_MULTI_PACK_HDR* hdr, U8 pack_type);
LOCAL void TRACE_cre_mpo_hdr(U32 len, TRACE_MULTI_PACK_HDR* hdr, U8 pack_type)
{
  hdr->core_no      = UTIL_get_cpu_core();
  hdr->len          = len + sizeof(TRACE_MULTI_PACK_HDR) +
                      sizeof(TRACE_MULTI_PACK_ID);
  hdr->proc_id      = 0;
  hdr->multi_pack   = TR_MSG_MULTIPKT & TRACE_LOW_NIBBLE;
  hdr->pack_typ     = pack_type & TRACE_LOW_NIBBLE;
}


/**
 * Check if specific class/level is enabled
 *
 * \parm clas   Trace class
 * \parm level  Trace level
 * \parm c_cnfg Pointer to trace class configuration info
 *
 * return \li TR_ENABLED_CLASS_ALL if all class/level enabled
 *                                 else specific level info
 */
LOCAL TRACE_clas_ena_stat TRACE_isclassenabled(U16 clas, TRACE_level level);
LOCAL TRACE_clas_ena_stat TRACE_isclassenabled(U16 clas, TRACE_level level)
{
  TRACE_clas_ena_stat status = TR_ENABLED_CLASS_NONE;

  status = (TRACE_clas_ena_stat)TR_core_bIsClassSelected2(clas, level);
  return status;
}

/**
 * Create timestamp message
 *
 * \parm msg    Pointer to message
 * \parm time_stmp timestamp mode
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
LOCAL ER TRACE_cre_timestamp_msg(TRACE_TIMESTAMP_MSG* msg,
                                 TRACE_TIMESTAMP_MODE time_stmp);
LOCAL ER TRACE_cre_timestamp_msg(TRACE_TIMESTAMP_MSG* msg,
                                 TRACE_TIMESTAMP_MODE time_stmp)
{
  ER       rc        = E_OK;
  U32      sz        = 0;  
  struct timespec tm = {0, 0};
  U64 tim_ms         = 0;
  TRACE_TIMESTAMP_64B systim = {0, 0};

  rc = clock_gettime(CLOCK_MONOTONIC, &tm);

  if(rc == E_OK)
  {
    /* get time in millisecs ! */
    tim_ms  = tm.tv_sec*1000;
    tim_ms += tm.tv_nsec/(1000*1000);
    systim.hi = tim_ms >>32;
    systim.lo = (U32)tim_ms;    
      
    switch(time_stmp)
    {
      case TRACE_TIMESTAMP_32BIT:
        sz = sizeof(TRACE_TIMESTAMP_32B);
        TRACE_pack_32bit_in_bigendian((U32)systim.lo,
                (U8*)(VP)&msg->payload.tm_32b.time);

        break;

      case TRACE_TIMESTAMP_64BIT:
        sz = sizeof(TRACE_TIMESTAMP_64B);
        TRACE_pack_32bit_in_bigendian((U32)systim.lo,
                 (U8*)(VP)&msg->payload.tm_64b.lo);
        TRACE_pack_32bit_in_bigendian((U32)systim.hi,
                 (U8*)(VP)&msg->payload.tm_64b.hi);
        break;

      case TRACE_TIMESTAMP_RTC:
        sz = sizeof(TRACE_RTC);
        /* FixMe: calculate RTC timestamp */
        break;

      case TRACE_TIMESTAMP_DISABLE:
        rc = E_FAIL;
        break;
      default:
        rc = E_FAIL;
        break;
    }
  }

  if(rc == E_OK)
  {
    /* Fill header for timestamp msg */
    TRACE_cre_hdr(sz, (U16)TR_CLASS_TRACE, (U8)TRACE_TIMESTAMP,
                  &msg->hdr, (U8)TR_MSG_NORMAL);
  }
  /* SWGII-6255 (SWGII-6232)Trace hangs when printout invalid frames (with timestamps) */
  else /*Either clock_gettime() failed or time_stmp is disabled or invalid */
  {
    systim.hi = 0; /* Since clock_gettime() failed, put time value as 0 */
    systim.lo = 0; /* Since clock_gettime() failed, put time value as 0 */

    switch(time_stmp)
    {
      case TRACE_TIMESTAMP_32BIT:
        sz = sizeof(TRACE_TIMESTAMP_32B);
        TRACE_pack_32bit_in_bigendian((U32)systim.lo,
                (U8*)(VP)&msg->payload.tm_32b.time);

        break;

      case TRACE_TIMESTAMP_64BIT:
        sz = sizeof(TRACE_TIMESTAMP_64B);
        TRACE_pack_32bit_in_bigendian((U32)systim.lo,
                 (U8*)(VP)&msg->payload.tm_64b.lo);
        TRACE_pack_32bit_in_bigendian((U32)systim.hi,
                 (U8*)(VP)&msg->payload.tm_64b.hi);
        break;
     default:
        break;      
    }
    /* Fill header for timestamp msg */
    TRACE_cre_hdr(sz, (U16)TR_CLASS_TRACE, (U8)TRACE_TIMESTAMP,
                  &msg->hdr, (U8)TR_MSG_NORMAL);
        
  }
  return rc;
}


/**
 * Register application for notification request
 *
 * \parm msg    Pointer to message
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
EXPORT ER TRACE_reg_notify_evt(TRACE_notify* msg)
{
  ER rc                 = E_OK;
  S32 cur_idx           = 0;
  S32 res               = 0;
  TRACE_notify_inf* inf = NULL;
  S32 data              = 0;
  TRACE_notify_evt evt  = msg->evt_id;
  S32 default_idx       = 0;

  msg = msg;
  if(msg != NULL)
  {
    if(evt >= TRACE_MAX_NOTIFY_EVT)
    {
      rc = E_FAIL;
    }

    if((E_FAIL != rc) &&
    (evt < TRACE_MAX_NOTIFY_EVT) &&
    (msg->func != NULL)&& (g_TRACE_mgr != NULL) &&
    (g_TRACE_mgr->sh != NULL)&&
    (g_TRACE_mgr->notify[evt].inf != NULL))
    {
      inf = g_TRACE_mgr->notify[evt].inf;
      cur_idx = UTIL_swap((VP)&g_TRACE_mgr->sh->notify[evt].idx,
                            g_TRACE_mgr->sh->notify[evt].idx + 1);
        if(cur_idx <= g_TRACE_mgr->sh->notify[evt].max)
        {
          res = UTIL_test_swap32((VP)&inf->stat,0x00,0x01);
          if(res == TRUE)
          {
            /* do registration for event */
              inf[cur_idx].prc_id = getpid();
              inf[cur_idx].fp = msg->func;

              /* start the notify stub thread 
               * for each process, this is created only once 
               * One stub thread handles(delivers)all kinds of events
               * Hence status about stub thread is kept only in index 0
               */         
              res = UTIL_test_swap32(&g_TRACE_mgr->notify[default_idx].stub.bCreatStub, 
                                                                      0x00, 0x01);
              if(res == TRUE)
              {
                if(E_OK == TRACE_create_task(&g_TRACE_mgr->notify[default_idx].stub.tsk_id.tid, 
                                     &TRACE_notify_stubtsk, (VP)g_TRACE_mgr) )
                {
                  g_TRACE_mgr->notify[default_idx].stub.tsk_id.exist = TRUE;
                }
              }
              rc = E_OK;
          }

          else
          {
            /* FixMe : error already registered */
          }
        }
        else
        {
          (void)UTIL_swap((VP)&g_TRACE_mgr->sh->notify[evt].idx,
                              g_TRACE_mgr->sh->notify[evt].max);
          rc = E_FAIL;
        }
    }
  }
  if((E_OK == rc) && (msg != NULL) && 
    (g_TRACE_mgr != NULL) && (g_TRACE_mgr->sh != NULL))
  {
    /* update current status via registered callback 
     * invoke the callback directly as we are in the same address space 
     */
    data = (msg->evt_id == (TRACE_notify_evt)TRACE_APP_FILTER)?
                ((S32)TRACE_get_appfilter_status(g_TRACE_mgr)):
                (S32)g_TRACE_mgr->sh->trc.d_cnfg.flgs.trip;

    (msg->func)((VP)&data);
  }
  return rc;
}

/**
 * UnRegister application for notification request
 *
 * \parm msg    Pointer to message
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
EXPORT ER TRACE_unreg_notify_evt(TRACE_notify* msg)
{
  ER rc                 = E_OK;
  S32 cur_idx           = 0;
  pid_t prc_id          = 0;
  S32 i                 = 0;
  S32 resid             = 0;
  BOOL bFound           = FALSE;
  TRACE_notify_inf* inf = NULL;
  TRACE_notify_evt evt  = msg->evt_id;

  resid = resid;/*To avoid comipler warning*/
  prc_id = getpid();
  if(msg != NULL)
  {
    if(evt >= TRACE_MAX_NOTIFY_EVT)
    {
      rc = E_FAIL;
    }
    if((E_OK == rc) &&
      (evt < TRACE_MAX_NOTIFY_EVT) &&
      (msg->func != NULL) &&
      (g_TRACE_mgr->notify[evt].inf != NULL) &&
      (g_TRACE_mgr->sh  != NULL))
    {
      inf = g_TRACE_mgr->notify[evt].inf;
      cur_idx = g_TRACE_mgr->sh->notify[evt].idx;
      while((i < cur_idx) && (bFound == FALSE))
      {
        if((inf[i].stat != 0) &&
          (inf[i].prc_id == prc_id)&&
          (inf[i].fp == msg->func))
        {
          (void)UTIL_swap((VP)&inf[i].stat, 0);
          bFound = TRUE;
        }
        i++;
      }
      if(bFound == FALSE)
      {
        /*FixMe: some kind of error logging */
        rc = E_FAIL;
      }
    }
  }
  else
  {
    /* parameter error */
    rc = E_FAIL;
  }
  return rc;
}

/**
 * Traceout proxy message
 *
 * \parm msg    Pointer to message
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
LOCAL ER TRACE_out_prxy_msg(TRACE_TRACEOUT_MSG* msg);
LOCAL ER TRACE_out_prxy_msg(TRACE_TRACEOUT_MSG* msg)
{
  ER rc = E_OK;

  /* Write to proxy buffer */
  if(g_TRACE_mgr->sh->prxy.status == TRUE)
  {
    rc = TRACE_q_push(&g_TRACE_mgr->prxy.buf, msg);
  }
  return rc;
}

/**
 * Register callback
 *
 * \parm clbk Pointer to trace callbk structure
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
LOCAL ER TRACE_callback_reg(TRACE_callbk* clbk);
LOCAL ER TRACE_callback_reg(TRACE_callbk* clbk)
{

  ER  rc                      = E_FAIL;
  S32 res                     = 0; 
  /*SWGIII-6802 Null pointer exception fix */
  if(g_TRACE_mgr!=NULL)
  {
      TRACE_callback_mgr* cmgr    = g_TRACE_mgr->chan.cmgr;

      if((clbk != NULL )&&(clbk->func != NULL))
      {
           if((clbk->chnl_id > NO_CHAN_SELECTED) &&
           (clbk->chnl_id < TR_LAST_LAUNCH_CHAN))
           {
               res = UTIL_test_swap32(&cmgr[clbk->chnl_id].chn_id,
                                            0x00, clbk->chnl_id);
                 if(res == TRUE)
                 {
                       cmgr[clbk->chnl_id].prc_id = getpid();
                       cmgr[clbk->chnl_id].fp = clbk->func;

                      /* start the channel stub thread 
          	     * for each process, this is created only once */
                     
                       res = UTIL_test_swap32(&g_TRACE_mgr->chan.stub.bCreatStub, 0x00, 0x01);
                       if(res == TRUE)
                       {                      
                           if(E_OK == TRACE_create_task(&g_TRACE_mgr->chan.stub.tsk_id.tid, 
                                     &TRACE_chnl_stubtsk, (VP)g_TRACE_mgr) )
                           {
                                 g_TRACE_mgr->chan.stub.tsk_id.exist = TRUE;
                           }
                       }
                      rc = E_OK;          
                 }  
                 else
                 {  
                     rc = E_FAIL;
                 }
         }  
         else
         {
             rc =  E_FAIL;
         }
    }
  }
  return rc;
}

/**
 * Unregister callback
 *
 * \parm clbk Pointer to trace callbk structure
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
LOCAL ER TRACE_callback_unreg(TRACE_callbk* clbk);
LOCAL ER TRACE_callback_unreg(TRACE_callbk* clbk)
{
  ER  rc   = E_FAIL;

  if(clbk != NULL)
  {
    if((clbk->chnl_id > NO_CHAN_SELECTED) &&
       (clbk->chnl_id < TR_LAST_LAUNCH_CHAN) &&
       (clbk->chnl_id == g_TRACE_mgr->chan.cmgr[clbk->chnl_id].chn_id))
    {
      (void) UTIL_swap(&g_TRACE_mgr->chan.cmgr[clbk->chnl_id].chn_id,
                       NO_CHAN_SELECTED);
	  g_TRACE_mgr->chan.cmgr[clbk->chnl_id].fp = NULL;
      rc = E_OK;
    }
  }
  else
  {
    rc = E_FAIL;
  }
  return rc;
}

/**
 * External interface for registering channel callback
 *
 * \parm  chan_id        Trace Channel ID
 *        p_Func         Callback func
 * return \li BOOL       TRUE, if registered successfully
 *                       FALSE, otherwise
 */
EXPORT BOOL TR_chan_acess_bRegChan(TR_tenTraceChan chan_id,
                                   TRACE_CALLBACK p_Func)
{
  ER                 rc     = E_OK;
  TRACE_callbk calbck_inf   = {0, NULL};

  calbck_inf.chnl_id = (TRACE_chnl_id)chan_id;
  calbck_inf.func    = p_Func;
/*SWGIII-3621: TRACE static constructor for init() dangerous */
  rc = TRACE_init_fp();
  if(rc >= E_OK)
  {
     rc = TRACE_callback_reg(&calbck_inf);
  }
  else
  {
     /*SWGIII-3621: TRACE static constructor for init() dangerous */
     TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "TR_chan_acess_bRegChan:TRACE_init failed\n");
  }
  return (rc >= E_OK) ? TRUE : FALSE;
}
/**
 * External interface for Unregistering channel callback
 *

 * \parm  chan_id        Trace Channel ID
 *        p_Func         Callback func
 * return \li BOOL       TRUE, if unregistered successfully
 *                       FALSE, otherwise
 */
EXPORT BOOL TR_chan_acess_bUnRegChan(TR_tenTraceChan chan_id,
                                     TRACE_CALLBACK p_Func)
{
  ER                 rc     = E_OK;
  TRACE_callbk calbck_inf   = {0, NULL};

  calbck_inf.chnl_id = (TRACE_chnl_id)chan_id;
  calbck_inf.func    = p_Func;
/*SWGIII-3621: TRACE static constructor for init() dangerous */
  rc = TRACE_init_fp();
  if(rc >= E_OK)
  {
     rc = TRACE_callback_unreg(&calbck_inf);
  }
  else
  {
     /*SWGIII-3621: TRACE static constructor for init() dangerous */
     TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "TR_chan_acess_bUnRegChan:TRACE_init failed\n");
	 
  }
  return (rc >= E_OK) ? TRUE : FALSE;
}


/**
 * Trace out normal message
 *
 * \parm msg    Pointer to message
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
LOCAL ER TRACE_out_normal_msg(TRACE_TRACEOUT_MSG* msg);
LOCAL ER TRACE_out_normal_msg(TRACE_TRACEOUT_MSG* msg)
{
  ER rc = E_OK;

  msg = msg;
  /* Get trace buffer fill status */
  msg->pkt.trc_msg.out_pkt.hdr.state_byte =
  TRACE_q_fill_status(&g_TRACE_mgr->trc.buf);

  /* Write to trace buffer */
  rc = TRACE_q_push(&g_TRACE_mgr->trc.buf, msg);
  return rc;
}

/**
 * Traceout large message
 *
 * \parm msg    Pointer to Trace manager
 * \parm app_msg Pointer to trace trace msg structure
 * \parm isp     Pointer to ISP header
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
LOCAL ER TRACE_out_large_msg(TRACE_mgr* mgr, TRACE_lb_msg* app_msg,
                                               const TRACE_ISP_HDR* isp);
LOCAL ER TRACE_out_large_msg(TRACE_mgr* mgr, TRACE_lb_msg* app_msg,
                                               const TRACE_ISP_HDR* isp)
{
  TRACE_msg*            umsg         = (TRACE_msg*)app_msg;
  ER                    rc           = E_OK;
  ER                    rc_mutexLock = E_OK;
  U32                   tx_pkt_cnt   = 1;
  TRACE_TRACEOUT_MSG    omsg;
  U32                   clen         = sizeof(omsg.pkt.trc_msg.multi_s_pkt.payld);
  U32                   rlen         = umsg->len;
  U32                   i            = 0;
  TRACE_MULTI_PACK_ID*  mpid         = &omsg.pkt.trc_msg.multi_s_pkt.pack_id;
  TRACE_MULTI_PACK_HDR* hdr          = (VP)&omsg.pkt.trc_msg.multi_s_pkt.hdr;
  U8                    pack_type    = 0;
  U8*                   payld        = omsg.pkt.trc_msg.multi_s_pkt.payld;
  S32                     eop          = FALSE;
  U8                    chksum       = (U8)TRACE_TX_CHECKSUM;
  U8                    isp_sz       = 0;

  mgr = mgr;
  memset(&omsg, 0, sizeof(omsg));
  omsg.tm_stamp = TRACE_TIMESTAMP_DISABLE;  
  /* Calculate total number of multipack packets */
  if(rlen >= sizeof(omsg.pkt.trc_msg.multi_s_pkt.payld))
  {
    while(1)
    {
      rlen = ((rlen >= clen)? (rlen-clen) : (clen-rlen));
      if(rlen < sizeof(omsg.pkt.trc_msg.multi_e_pkt.payld))
      {
        clen = sizeof(omsg.pkt.trc_msg.multi_e_pkt.payld);
        tx_pkt_cnt++;
        break;
      }
      else
      {
        clen = sizeof(omsg.pkt.trc_msg.multi_m_pkt.payld);
        tx_pkt_cnt++;
      }
    }
    rlen = umsg->len;
    clen = (rlen >= sizeof(omsg.pkt.trc_msg.multi_s_pkt.payld))?
            sizeof(omsg.pkt.trc_msg.multi_s_pkt.payld) : rlen;
  }
  else
  {
    clen = rlen;
  }

  /* Construct Multipack identifier */
  mpid->id3 = (tx_pkt_cnt & TRACE_MP_ID1_MASK);
  mpid->id2 = (tx_pkt_cnt & TRACE_MP_ID2_MASK) >> TRACE_SHIFT_8bits;
  mpid->id1 = (tx_pkt_cnt & TRACE_MP_ID3_MASK) >> TRACE_SHIFT_16bits;

  /* Construct multipack header */
  (void)TRACE_cre_mpos_hdr(clen, (U16)umsg->clas, (U8)umsg->level,
                           hdr, (U8)TR_MSG_MULTIPKT, (U8)TR_MSG_MULTI_START);

  /* Construct multipack payload */
  if(NULL != isp)
  {
    isp_sz = sizeof(TRACE_ISP_HDR);
    memcpy((VP)payld, isp, isp_sz);
  }
  memcpy((VP)(payld + isp_sz), (VP)umsg->msg, (clen - isp_sz));


  /* Prepare timestamp message */
  omsg.tm_stamp = g_TRACE_mgr->sh->trc.d_cnfg.flgs.timestamp;
  (void)TRACE_cre_timestamp_msg(&omsg.pkt.tm_msg, omsg.tm_stamp);

  tx_pkt_cnt--;

  TRACE_calculate_checksum(payld, (U8)clen, &chksum, (TRACE_checksum)chksum);
  if(tx_pkt_cnt == 0)
  {
    /* PRQA: Lint Message 661: out of bound not possible as the pointer "payId" is */
    /* PRQA: Lint Message 661: being reassigned to point to end pkt payload to accomodate checksum */
    /*lint -save -e661 */
    payld[clen] = chksum;
    /*lint -restore*/
    hdr->len++;
  }

  /* PRQA: Lint Message 454: If and only if mutex_lock is success, we are unlocking. Lint is not checking */
  /* PRQA: Lint Message 454: what is the if() condition  but only checking whether unlock is called inside if() */
  /*lint -save -e454 */
  /* PRQA: Lint Message 456: If and only if mutex_lock is success, we are unlocking. Lint is not checking */
  /* PRQA: Lint Message 456: what is the if() condition  but only checking whether unlock is called inside if() */
  /*lint -save -e456 */
  rc_mutexLock = TRACE_mutex_lock(&mgr->sh->trc.lrgmsg_lock);

  if(rc_mutexLock == E_OK) /* Lock successfull*/
  {
    rc = TRACE_q_push(&mgr->trc.buf, &omsg);


  if((rc >= E_OK) && (tx_pkt_cnt > 0))
  {
    /* Multipack START packet already transmitted */
    tx_pkt_cnt = 1;

    /* Increment read index for user buffer */
    i += (clen - isp_sz);

    /* Update pending read length */
    rlen = (umsg->len - isp_sz) - i ;

    do
    {
      if(rlen < sizeof(omsg.pkt.trc_msg.multi_e_pkt.payld))
      {
        mpid      = &omsg.pkt.trc_msg.multi_e_pkt.pack_id;
        hdr       = &omsg.pkt.trc_msg.multi_e_pkt.hdr;
        pack_type = TR_MSG_MULTI_END;
        payld     = omsg.pkt.trc_msg.multi_e_pkt.payld;
        clen      = rlen;
        eop       = TRUE;
      }
      else
      {
        mpid      = &omsg.pkt.trc_msg.multi_m_pkt.pack_id;
        hdr       = &omsg.pkt.trc_msg.multi_m_pkt.hdr;
        pack_type = TR_MSG_MULTI_MIDDLE;
        payld     = omsg.pkt.trc_msg.multi_m_pkt.payld;
        clen      = (rlen >= sizeof(omsg.pkt.trc_msg.multi_m_pkt.payld))?
                    sizeof(omsg.pkt.trc_msg.multi_m_pkt.payld) : rlen;
      }

      /* Increment multipack ID count */
      tx_pkt_cnt++;

      /* Construct Multipack identifier */
      mpid->id3 = (tx_pkt_cnt & TRACE_MP_ID1_MASK);
      mpid->id2 = (tx_pkt_cnt & TRACE_MP_ID2_MASK) >> TRACE_SHIFT_8bits;
      mpid->id1 = (tx_pkt_cnt & TRACE_MP_ID3_MASK) >> TRACE_SHIFT_16bits;

      /* Construct multipack header */
      (void)TRACE_cre_mpo_hdr(clen, hdr, pack_type);

      /* Construct multipack payload */
      memcpy((VP)payld, (VP)&umsg->msg[i], clen);
      TRACE_calculate_checksum(payld, (U8)clen, &chksum,(TRACE_checksum)chksum);

      if(eop == TRUE)
      {
        /* Insert checksum */
        /* PRQA: Lint Message 661: out of bound not possible as the pointer "payId" is being reassigned */
        /* PRQA: Lint Message 661: to point end pkt payload */
        /*lint -save -e661 */
        payld[clen] = chksum;
        /*lint -restore */
        hdr->len++;
      }
      else
      {
        /* Increment read index for user buffer */
        i += clen;

        /* Update pending read length */
        rlen = (umsg->len - isp_sz) - i ;
      }

      /* Prepare timestamp message */
      omsg.tm_stamp = mgr->sh->trc.d_cnfg.flgs.timestamp;
      (void)TRACE_cre_timestamp_msg(&omsg.pkt.tm_msg, omsg.tm_stamp);

      /* Transmit packet */
      rc = TRACE_q_push(&mgr->trc.buf, &omsg);

    }while((eop == FALSE) && (rc >= E_OK));
  }
  /* Deleted if(rc == E_OK) check since mutex that has been locked to be unlocked*/
  TRACE_mutex_unlock(&mgr->sh->trc.lrgmsg_lock);
 }
      else
     {
       rc = rc_mutexLock;
  }
  return rc;
  /*lint -restore */
  /*lint -restore */
}

LOCAL ER TRACE_io_traceout_lb_msg(TRACE_lb_msg* msg);
LOCAL ER TRACE_io_traceout_lb_msg(TRACE_lb_msg* msg)
{
  TRACE_ISP_HDR  isp_inf  = {0, 0, 0, 0};
  ER             rc       = E_OK;
  U16             isp_cmd = (U16)TRACE_ISP_CMD_DATA;

  if(TR_COMP_TESTER == msg->clas)
  {
    /* only for BP add class id, otherwise compid = trace class id */
    msg->clas = msg->clas + TR_CLASS_MST_TESTER;
  }

  TRACE_pack_16bit_in_bigendian(isp_cmd,(VP)&isp_cmd);
  isp_inf.cmd = isp_cmd;
  TRACE_pack_32bit_in_bigendian(msg->sock_id,(VP)&isp_inf.sock);
  msg->len += sizeof(TRACE_ISP_HDR);

  rc = TRACE_out_large_msg(g_TRACE_mgr, msg, &isp_inf);

  return rc;
}


/**
 * Traceout message
 *
 * \parm app_msg Pointer to structure carrying user passed data.
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
LOCAL ER TRACE_io_traceoutmsg(TRACE_msg* app_msg);
LOCAL ER TRACE_io_traceoutmsg(TRACE_msg* app_msg)
{
  TRACE_TRACEOUT_MSG   msg;
  S32                  actv         = TR_ENABLED_CLASS_NONE;
  ER                   rc           = E_OK;
  TRACE_status_msg     st_msg       = {0, {0}};
  U8                   len          = sizeof(st_msg.status);
  TRACE_TIMESTAMP_MODE tm_mode      = TRACE_TIMESTAMP_DISABLE;  
  memset(&msg, 0, sizeof(TRACE_TRACEOUT_MSG));
  msg.tm_stamp = TRACE_TIMESTAMP_DISABLE;

  /*SWGIII-6802 Null pointer exception fix */
  if((app_msg != NULL) && (app_msg->msg !=NULL) &&
    ((app_msg->len > 0) && (app_msg->len <= TRACE_LARGE_MSG_SZ))) /* Fix for SWGII-5735*/
  {
    actv = TRACE_isclassenabled((U16)app_msg->clas, app_msg->level);

    if(actv != TR_ENABLED_CLASS_NONE)
    {

      if(app_msg->len > TRACE_NORMAL_MSG_SZ)
      {
        /* @dhd3kor<24.02.2011>: fixing lint warning 
         * Convert the message into large message format and send the message
         * to large msg traceout handler.
         */
        TRACE_lb_msg lrg_msg;
        memset(&lrg_msg, 0, sizeof(lrg_msg) );
        lrg_msg.len   = app_msg->len;
        lrg_msg.clas  = app_msg->clas;
        lrg_msg.level = app_msg->level;
        lrg_msg.msg   = app_msg->msg;
        lrg_msg.sock_id = 0; /*invalid socket id*/
        
        rc = TRACE_out_large_msg(g_TRACE_mgr, &lrg_msg, NULL);
      }
      else
      {
        (void)TRACE_cre_hdr(app_msg->len, (U16)app_msg->clas, (U8)app_msg->level,
                            &msg.pkt.trc_msg.out_pkt.hdr,
                            (U8)TR_MSG_NORMAL);

        (void)memcpy((VP)&msg.pkt.trc_msg.out_pkt.payld,
                     (VP)app_msg->msg, app_msg->len);


        /* Prepare timestamp msg */
        if((TR_ENABLED_CLASS_PMM == ((U32)actv & TR_ENABLED_CLASS_PMM)) ||
          (TRACE_TIMESTAMP_DISABLE != g_TRACE_mgr->sh->trc.d_cnfg.flgs.timestamp)||
          (TRACE_TIMESTAMP_DISABLE != g_TRACE_mgr->sh->prxy.timestamp_mode))
        {
          /* PMM message always has timestamp 
           * RBCM request: FATAL messages should not have timestamp by default
           * when enabled timestamp is sent otherwise not.
           * @kbm2kor: Fix:SWGII-4043
           */ 
          tm_mode = (g_TRACE_mgr->sh->trc.d_cnfg.flgs.timestamp == TRACE_TIMESTAMP_DISABLE)?
                    TRACE_TIMESTAMP_32BIT: g_TRACE_mgr->sh->trc.d_cnfg.flgs.timestamp;

          (void)TRACE_cre_timestamp_msg(&msg.pkt.tm_msg, tm_mode);
        }

        if(TR_ENABLED_CLASS_PXY == ((U32)actv & TR_ENABLED_CLASS_PXY))
        {
          /* Send to Proxy */
          msg.tm_stamp =(TRACE_TIMESTAMP_MODE)g_TRACE_mgr->sh->prxy.timestamp_mode;
          TRACE_out_prxy_msg(&msg);
        }

        if(TR_ENABLED_CLASS_TRC == ((U32)actv & TR_ENABLED_CLASS_TRC))
        {
          /* Send to TTFis */
          msg.tm_stamp = g_TRACE_mgr->sh->trc.d_cnfg.flgs.timestamp;
          rc = TRACE_out_normal_msg(&msg);
        }
      }
    }
    else
    {
       /*Fix for SWGIIX-272 */
       rc = E_FAIL;
    }
  }
  else
  {
    if(app_msg == NULL)
    {
      rc = E_FAIL;
    }
    else
    {
      /* send error message to TTFis */
      st_msg.status = (U8)TRACE_NO_SPT;
      st_msg.payld[TRACE_INDEX_0] = (U8)TRACE_INVALID_MSG_SZ;
      len += TRACE_INDEX_1;
      (void)TRACE_snd_status(g_TRACE_mgr,(U8*)(VP)&st_msg,len);
    }
  }
  return rc;
}

/**
 * This function is added for backward compatibility
 * External interface for trace out messages to Proxy
 *
 * \parm  uwLen       Length of the msg in bytes
 *        trClass     Trace class of the msg
 *        trLevel     Trace level of the msg
 *        p_ubTrData  ptr to a byte buffer containing msg
 * return \li U32      number of bytes traced out
 */
EXPORT U32 TR_proxy_uwTrace(U32 uwLen, TR_tenTraceClass trClass,
                      TR_tenTraceLevel trLevel, U8* p_ubTrData)
{
  ER          rc = E_OK;
  TRACE_msg   msg;
  
  memset(&msg, 0, sizeof(msg) );
  msg.len   = uwLen;
  msg.clas  = (U32)trClass;
  msg.level = trLevel;
  msg.msg   = (VP)p_ubTrData;
/*SWGIII-3621: TRACE static constructor for init() dangerous */
  rc = TRACE_init_fp();
  if (rc >= E_OK)
  {
     rc = TRACE_io_traceoutmsg(&msg);
  }
  else
  {
      /*SWGIII-3621: TRACE static constructor for init() dangerous */
     TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "TR_proxy_uwTrace :TRACE_init failed\n");
  }
  return (rc >= E_OK) ? uwLen : E_OK;
}

/**
 * External interface for trace out messages
 *
 * \parm  uwLen       Length of the msg in bytes
 *        trClass     Trace class of the msg
 *        trLevel     Trace level of the msg
 *        p_ubTrData  ptr to a byte buffer containing msg
 * return \li U32      number of bytes traced out
 */

EXPORT U32 TR_core_uwTraceOut(U32 uwLen, U16 trClass, TR_tenTraceLevel trLevel,
                             U8* p_ubTrData)
{
  ER        rc = E_OK;
  TRACE_msg msg;

  memset(&msg, 0, sizeof(msg) );
  msg.len   = uwLen;
  msg.clas  = trClass;
  msg.level = trLevel;
  msg.msg   = (VP)p_ubTrData;
/*SWGIII-3621: TRACE static constructor for init() dangerous */
  rc = TRACE_init_fp();
  if (rc >= E_OK)
  {
     rc = TRACE_io_traceoutmsg(&msg);
  }
  else
  {
      /*SWGIII-3621: TRACE static constructor for init() dangerous */
     TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "TR_core_uwTraceOut:TRACE_init failed\n");
  }
  return (rc >= E_OK) ? uwLen : E_OK;

}

/**
 * This function is to provide the current block mode status
 *
 * \parm  p_u32	TraceBlockMode
 *		p_u32	ProxyBlockMode
 *        
 * return \li 	void
 *                       
 */
EXPORT void TR_get_blockmode_status(U32 *TraceBlockMode,U32 *ProxyBlockMode)
{
    if(g_TRACE_mgr != NULL)
    {
		if (( TraceBlockMode != NULL ) && ( ProxyBlockMode != NULL ))
		{
			*TraceBlockMode = (TRACE_BLOCK_MODE == g_TRACE_mgr->sh->trc.buf.blck_mode)? TRUE : FALSE;
			*ProxyBlockMode = (TRACE_BLOCK_MODE == g_TRACE_mgr->sh->prxy.buf.blck_mode)? TRUE : FALSE;
		}
    }
	else
	{
		TRACE_SYSLOG(TRACE_SYSLOG_ERROR ,"g_TRACE_mgr is NULL\n");
	}
}
/**
 * External interface for trace out MST messages
 *
 * \parm  uwCompId        Component ID of the MST tester class
 *        uwSockId        socketID as received from MST client
 *        uwBufferLen     Length of the msg in bytes
 *        p_ubDataBuffer  ptr to a byte buffer containing msg
 * return \li U32          number of bytes traced out
 */
EXPORT U32 TR_core_uwTraceBinOutput(U32 uwCompId, U32 uwSockId, U32 uwBufferLen,
                                   U8* p_ubDataBuffer)
{
  ER                rc = E_OK;
  TRACE_lb_msg  lb_msg;

  memset(&lb_msg, 0, sizeof(lb_msg) );
  lb_msg.len     = uwBufferLen;
  lb_msg.msg     = (VP)p_ubDataBuffer;
  lb_msg.clas    = uwCompId;
  lb_msg.level   = TR_LEVEL_USER_4;
  lb_msg.sock_id = uwSockId;
/*SWGIII-3621: TRACE static constructor for init() dangerous */
  rc = TRACE_init_fp();
  if (rc >= E_OK)
  {
    rc = TRACE_io_traceout_lb_msg(&lb_msg);
  }
  else
  {
   /*SWGIII-3621: TRACE static constructor for init() dangerous */
    TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "TR_core_uwTraceBinOutput:TRACE_init failed\n");
  }
  return (rc >= E_OK) ? uwBufferLen : E_OK;
}

/* In Linux there should be no dependency from FE to impl, so defined the *
 * below API also in trace_impl.c for Linux alone temporarily             */

/**
 * Provide shared mem pointer to BE
 * 
 * \parm  mem    ptr to TRACE__sh_mem
 * return \li ER E_OK if assigned successfully
 *               E_FAIL, otherwise
 */
EXPORT ER TRACE_get_sh_mem(TRACE_sh_mem* mem)
{
  ER rc = E_FAIL;
  if(mem != NULL)
  {
    if(g_TRACE_mgr != NULL)
    {
      mem->p_shmem = (U8*)g_TRACE_mgr->sh;
      rc = E_OK;
    }
  }
  return rc;
}

EXPORT ER TRACE_media_detach(TRACE_media_type mt)
{
  ER  rc    = E_OK;
  U32 event = (U32)TRACE_PROXY_MEDIA_DETACH;

  mt = mt;
/*SWGIII-3621: TRACE static constructor for init() dangerous */
  rc = TRACE_init_fp();
  if (rc >= E_OK)
  {
  /*SWGIII-6802 Null pointer exception fix */
    if(g_TRACE_mgr != NULL)
    {
       rc = TRACE_set_flag(&g_TRACE_mgr->sh->evt_id[EVT_PRXY_Q], event);
    }
  }
  else
  {
    /*SWGIII-3621: TRACE static constructor for init() dangerous */
      TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "TRACE_media_detach:TRACE_init failed\n");
  }
  return rc;
}


EXPORT ER TRACE_media_attach(char *path)
{
  ER  rc    = E_OK;
  U32 event = (U32)TRACE_PROXY_MEDIA_ATTACH;

/*SWGIII-3621: TRACE static constructor for init() dangerous */
  rc = TRACE_init_fp();
  if(path != NULL)
  {
    size_t len = strlen(path);
    if(len < (TRACE_STORAGE_DEV_LEN + TRACE_FILE_NM_LEN))
    {
        strncpy((char *)g_TRACE_mgr->sh->prxy.cfg_file_path,path,len);
    }
  }
  else
  {
    TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Proxy: path is null\n");
  }
  if (rc >=E_OK)
  {
     rc = TRACE_set_flag(&g_TRACE_mgr->sh->evt_id[EVT_PRXY_Q], event);
  }
  else
  {
   /*SWGIII-3621: TRACE static constructor for init() dangerous */
     TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "TRACE_media_attach:TRACE_init failed\n");

  }

  return rc;
}

/**
 * Initialize Trace frontend.
 *
 * \parm None
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
EXPORT ER TRACE_frontend_init(void)
{
    return TRACE_start();  /*  SWGIIX-2071: CommandLine parameter not recieved in trace be */

}

/**
 * Trace entry point 
 *
 * \parm ac Startup or shutdown info, argument count in startup case
 * \parm av If av[0] is present (ac >= 1) it will be treated as BOOL* and used
 *          to store the information whether the backend was started or not.
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
EXPORT ER TRACE_main(S32 ac, U8** av)
{
  ER rc  = E_OK;
  av = av;
  if(ac >= 0)
  {
      rc = TRACE_frontend_init();
  }
  else
  {
    if(NULL != g_TRACE_mgr)
    {
      rc = TRACE_stop(&g_TRACE_mgr);
    }
  }
  return rc;
}

/**
 * "Backward Compatibility" wrapper for trace frontend startup
 */
EXPORT ER trace_drv_core_init (void)
{
  ER rc = TRACE_main(0, NULL);
  return rc;
}

/**
 * "Backward Compatibility" wrapper for trace shutdown
 */
EXPORT ER trace_drv_core_uninit(void)
{
  ER rc = TRACE_main(-1, NULL);
  return rc;
}


/***********************************************************************
 * \Func: TR_core_SendCmd
 * 
 * This function enables the application developer to request
 * execution of trace commands from application without using TTFis. 
 * 
 * \Param:	buf   Pointer to the buffer containing command information 
 *                and data.
 *          size  Size of the buffer passed.
 *
 * \Return: ER		E_OK     upon normal completion
  *                 E_FAIL    In case buffer pointer passed as argument is NULL.
 **********************************************************************/
EXPORT S32 TR_core_s32SendCmd(void* buf, U32 size)
{
    ER rc = E_OK;
    U32 event = (U32)TRACE_SIG_SEND_CMD_TSK;
   /*SWGIII-3621: TRACE static constructor for init() dangerous */
    rc = TRACE_init_fp();
    if (rc >=E_OK)
    {
    /* validate input parameters */
    if(( NULL == buf) || (size <= 0) || 
        (size >= (sizeof(g_TRACE_mgr->sh->sndcmd.cmd_buf) - 0x02)))
    {
		rc = E_FAIL;
    }
    if(rc >= E_OK)
    {
      /* acquire lock */
      TRACE_mutex_lock(&g_TRACE_mgr->sh->sndcmd.lock);

      if(TRUE == g_TRACE_mgr->sh->sndcmd.cmd_cmplt)
      {
        /* trace BE had indicated that it has done with processing 
         * now it is safe to accept next command
         */
        g_TRACE_mgr->sh->sndcmd.cmd_cmplt = FALSE;

        /* copy the data into shared memory */
		if(buf != NULL)
		{
          memcpy((VP)&g_TRACE_mgr->sh->sndcmd.cmd_buf, (U8*)buf, size);
		}
         g_TRACE_mgr->sh->sndcmd.buf_len = size; 

      
               /* inform BE that command is now available for processing */
		 /*Set Flag to send command task in BE*/

          TRACE_set_flag(&g_TRACE_mgr->sh->evt_id[EVT_SND_CMD], event);
	  }
	  else
	  {
         /* application can retry again */
			rc = E_FAIL;
	  }
      /*Release Lock*/
      TRACE_mutex_unlock(&g_TRACE_mgr->sh->sndcmd.lock);
	}
	    
    }
    else
    {
      /*SWGIII-3621: TRACE static constructor for init() dangerous */
    	TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "TR_core_s32SendCmd:TRACE_init failed\n");
    }

    return rc;
  }
